//
// Licensed to the Apache Software Foundation (ASF) under one
// or more contributor license agreements.  See the NOTICE file
// distributed with this work for additional information
// regarding copyright ownership.  The ASF licenses this file
// to you under the Apache License, Version 2.0 (the
// "License"); you may not use this file except in compliance
// with the License.  You may obtain a copy of the License at
//
//   http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing,
// software distributed under the License is distributed on an
// "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
// KIND, either express or implied.  See the License for the
// specific language governing permissions and limitations
// under the License.
//

package com.cloud.hypervisor.kvm.resource.wrapper;

import org.libvirt.Connect;
import org.libvirt.LibvirtException;

import com.cloud.agent.api.Answer;
import com.cloud.agent.api.RebootAnswer;
import com.cloud.agent.api.RebootCommand;
import com.cloud.agent.api.to.VirtualMachineTO;
import com.cloud.hypervisor.kvm.resource.LibvirtComputingResource;
import com.cloud.resource.CommandWrapper;
import com.cloud.resource.ResourceWrapper;

@ResourceWrapper(handles =  RebootCommand.class)
public final class LibvirtRebootCommandWrapper extends CommandWrapper<RebootCommand, Answer, LibvirtComputingResource> {


    @Override
    public Answer execute(final RebootCommand command, final LibvirtComputingResource libvirtComputingResource) {
        final LibvirtUtilitiesHelper libvirtUtilitiesHelper = libvirtComputingResource.getLibvirtUtilitiesHelper();
        final VirtualMachineTO vmSpec = command.getVirtualMachine();

        try {
            final Connect conn = libvirtUtilitiesHelper.getConnectionByVmName(command.getVmName());
            final String result = libvirtComputingResource.rebootVM(conn, command.getVmName());
            if (result == null) {
                Integer vncPort = null;
                try {
                    vncPort = libvirtComputingResource.getVncPort(conn, command.getVmName());
                } catch (final LibvirtException e) {
                    logger.trace("Ignoring libvirt error.", e);
                }
                if (vmSpec != null) {
                    libvirtComputingResource.applyDefaultNetworkRules(conn, vmSpec, false);
                }
                return new RebootAnswer(command, null, vncPort);
            } else {
                return new RebootAnswer(command, result, false);
            }
        } catch (final LibvirtException e) {
            return new RebootAnswer(command, e.getMessage(), false);
        }
    }
}
