/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hc.client5.http.impl.cache;

import java.net.URI;
import java.net.URISyntaxException;
import java.time.Instant;
import java.util.ArrayList;
import java.util.HashSet;
import java.util.Iterator;
import java.util.Locale;
import java.util.Objects;
import java.util.Set;
import org.apache.hc.client5.http.cache.HttpCacheEntry;
import org.apache.hc.client5.http.cache.RequestCacheControl;
import org.apache.hc.client5.http.cache.ResponseCacheControl;
import org.apache.hc.client5.http.impl.cache.CacheConfig;
import org.apache.hc.client5.http.impl.cache.CacheKeyGenerator;
import org.apache.hc.client5.http.impl.cache.CacheSuitability;
import org.apache.hc.client5.http.impl.cache.CacheValidityPolicy;
import org.apache.hc.client5.http.utils.DateUtils;
import org.apache.hc.client5.http.validator.ETag;
import org.apache.hc.core5.http.Header;
import org.apache.hc.core5.http.HttpRequest;
import org.apache.hc.core5.http.MessageHeaders;
import org.apache.hc.core5.http.Method;
import org.apache.hc.core5.http.message.MessageSupport;
import org.apache.hc.core5.util.TimeValue;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

class CachedResponseSuitabilityChecker {
    private static final Logger LOG = LoggerFactory.getLogger(CachedResponseSuitabilityChecker.class);
    private final CacheValidityPolicy validityStrategy;
    private final boolean sharedCache;
    private final boolean staleifError;

    CachedResponseSuitabilityChecker(CacheValidityPolicy validityStrategy, CacheConfig config) {
        this.validityStrategy = validityStrategy;
        this.sharedCache = config.isSharedCache();
        this.staleifError = config.isStaleIfErrorEnabled();
    }

    CachedResponseSuitabilityChecker(CacheConfig config) {
        this(new CacheValidityPolicy(config), config);
    }

    public CacheSuitability assessSuitability(RequestCacheControl requestCacheControl, ResponseCacheControl responseCacheControl, HttpRequest request, HttpCacheEntry entry, Instant now) {
        boolean fresh;
        if (!this.requestMethodMatch(request, entry)) {
            LOG.debug("Request method and the cache entry method do not match");
            return CacheSuitability.MISMATCH;
        }
        if (!this.requestUriMatch(request, entry)) {
            LOG.debug("Target request URI and the cache entry request URI do not match");
            return CacheSuitability.MISMATCH;
        }
        if (!this.requestHeadersMatch(request, entry)) {
            LOG.debug("Request headers nominated by the cached response do not match those of the request associated with the cache entry");
            return CacheSuitability.MISMATCH;
        }
        if (!this.requestHeadersMatch(request, entry)) {
            LOG.debug("Request headers nominated by the cached response do not match those of the request associated with the cache entry");
            return CacheSuitability.MISMATCH;
        }
        if (requestCacheControl.isNoCache()) {
            LOG.debug("Request contained no-cache directive; the cache entry must be re-validated");
            return CacheSuitability.REVALIDATION_REQUIRED;
        }
        if (this.isResponseNoCache(responseCacheControl, entry)) {
            LOG.debug("Response contained no-cache directive; the cache entry must be re-validated");
            return CacheSuitability.REVALIDATION_REQUIRED;
        }
        if (this.hasUnsupportedConditionalHeaders(request)) {
            LOG.debug("Response from cache is not suitable due to the request containing unsupported conditional headers");
            return CacheSuitability.REVALIDATION_REQUIRED;
        }
        if (!this.isConditional(request) && entry.getStatus() == 304) {
            LOG.debug("Unconditional request and non-modified cached response");
            return CacheSuitability.REVALIDATION_REQUIRED;
        }
        if (!this.allConditionalsMatch(request, entry, now)) {
            LOG.debug("Response from cache is not suitable due to the conditional request and with mismatched conditions");
            return CacheSuitability.REVALIDATION_REQUIRED;
        }
        TimeValue currentAge = this.validityStrategy.getCurrentAge(entry, now);
        if (LOG.isDebugEnabled()) {
            LOG.debug("Cache entry current age: {}", (Object)currentAge);
        }
        TimeValue freshnessLifetime = this.validityStrategy.getFreshnessLifetime(responseCacheControl, entry);
        if (LOG.isDebugEnabled()) {
            LOG.debug("Cache entry freshness lifetime: {}", (Object)freshnessLifetime);
        }
        boolean bl = fresh = currentAge.compareTo(freshnessLifetime) < 0;
        if (!fresh && responseCacheControl.isMustRevalidate()) {
            LOG.debug("Response from cache is not suitable due to the response must-revalidate requirement");
            return CacheSuitability.REVALIDATION_REQUIRED;
        }
        if (!fresh && this.sharedCache && responseCacheControl.isProxyRevalidate()) {
            LOG.debug("Response from cache is not suitable due to the response proxy-revalidate requirement");
            return CacheSuitability.REVALIDATION_REQUIRED;
        }
        if (fresh && requestCacheControl.getMaxAge() >= 0L && currentAge.toSeconds() > requestCacheControl.getMaxAge() && requestCacheControl.getMaxStale() == -1L) {
            LOG.debug("Response from cache is not suitable due to the request max-age requirement");
            return CacheSuitability.REVALIDATION_REQUIRED;
        }
        if (fresh && requestCacheControl.getMinFresh() >= 0L && (requestCacheControl.getMinFresh() == 0L || freshnessLifetime.toSeconds() - currentAge.toSeconds() < requestCacheControl.getMinFresh())) {
            LOG.debug("Response from cache is not suitable due to the request min-fresh requirement");
            return CacheSuitability.REVALIDATION_REQUIRED;
        }
        if (requestCacheControl.getMaxStale() >= 0L) {
            long stale;
            long l = stale = currentAge.compareTo(freshnessLifetime) > 0 ? currentAge.toSeconds() - freshnessLifetime.toSeconds() : 0L;
            if (LOG.isDebugEnabled()) {
                LOG.debug("Cache entry staleness: {} SECONDS", (Object)stale);
            }
            if (stale >= requestCacheControl.getMaxStale()) {
                LOG.debug("Response from cache is not suitable due to the request max-stale requirement");
                return CacheSuitability.REVALIDATION_REQUIRED;
            }
            LOG.debug("The cache entry is fresh enough");
            return CacheSuitability.FRESH_ENOUGH;
        }
        if (fresh) {
            LOG.debug("The cache entry is fresh");
            return CacheSuitability.FRESH;
        }
        if (responseCacheControl.getStaleWhileRevalidate() > 0L) {
            long stale;
            long l = stale = currentAge.compareTo(freshnessLifetime) > 0 ? currentAge.toSeconds() - freshnessLifetime.toSeconds() : 0L;
            if (stale < responseCacheControl.getStaleWhileRevalidate()) {
                LOG.debug("The cache entry is stale but suitable while being revalidated");
                return CacheSuitability.STALE_WHILE_REVALIDATED;
            }
        }
        LOG.debug("The cache entry is stale");
        return CacheSuitability.STALE;
    }

    boolean requestMethodMatch(HttpRequest request, HttpCacheEntry entry) {
        return request.getMethod().equalsIgnoreCase(entry.getRequestMethod()) || Method.HEAD.isSame(request.getMethod()) && Method.GET.isSame(entry.getRequestMethod());
    }

    boolean requestUriMatch(HttpRequest request, HttpCacheEntry entry) {
        try {
            URI requestURI = CacheKeyGenerator.normalize(request.getUri());
            URI cacheURI = new URI(entry.getRequestURI());
            if (requestURI.isAbsolute()) {
                return Objects.equals(requestURI, cacheURI);
            }
            return Objects.equals(requestURI.getPath(), cacheURI.getPath()) && Objects.equals(requestURI.getQuery(), cacheURI.getQuery());
        }
        catch (URISyntaxException ex) {
            return false;
        }
    }

    boolean requestHeadersMatch(HttpRequest request, HttpCacheEntry entry) {
        Iterator<Header> it = entry.headerIterator("Vary");
        if (it.hasNext()) {
            HashSet headerNames = new HashSet();
            while (it.hasNext()) {
                Header header = it.next();
                MessageSupport.parseTokens((Header)header, e -> headerNames.add(e.toLowerCase(Locale.ROOT)));
            }
            ArrayList tokensInRequest = new ArrayList();
            ArrayList tokensInCache = new ArrayList();
            for (String headerName : headerNames) {
                if (headerName.equalsIgnoreCase("*")) {
                    return false;
                }
                CacheKeyGenerator.normalizeElements((MessageHeaders)request, headerName, tokensInRequest::add);
                CacheKeyGenerator.normalizeElements(entry.requestHeaders(), headerName, tokensInCache::add);
                if (Objects.equals(tokensInRequest, tokensInCache)) continue;
                return false;
            }
        }
        return true;
    }

    boolean isResponseNoCache(ResponseCacheControl responseCacheControl, HttpCacheEntry entry) {
        if (responseCacheControl.isNoCache()) {
            Set<String> noCacheFields = responseCacheControl.getNoCacheFields();
            if (noCacheFields.isEmpty()) {
                LOG.debug("Revalidation required due to unqualified no-cache directive");
                return true;
            }
            for (String field : noCacheFields) {
                if (!entry.containsHeader(field)) continue;
                if (LOG.isDebugEnabled()) {
                    LOG.debug("Revalidation required due to no-cache directive with field {}", (Object)field);
                }
                return true;
            }
        }
        return false;
    }

    public boolean isConditional(HttpRequest request) {
        return this.hasSupportedEtagValidator(request) || this.hasSupportedLastModifiedValidator(request);
    }

    public boolean allConditionalsMatch(HttpRequest request, HttpCacheEntry entry, Instant now) {
        boolean lastModifiedValidatorMatches;
        boolean hasEtagValidator = this.hasSupportedEtagValidator(request);
        boolean hasLastModifiedValidator = this.hasSupportedLastModifiedValidator(request);
        if (!hasEtagValidator && !hasLastModifiedValidator) {
            return true;
        }
        boolean etagValidatorMatches = hasEtagValidator && this.etagValidatorMatches(request, entry);
        boolean bl = lastModifiedValidatorMatches = hasLastModifiedValidator && this.lastModifiedValidatorMatches(request, entry, now);
        if (hasEtagValidator && hasLastModifiedValidator && (!etagValidatorMatches || !lastModifiedValidatorMatches)) {
            return false;
        }
        if (hasEtagValidator && !etagValidatorMatches) {
            return false;
        }
        return !hasLastModifiedValidator || lastModifiedValidatorMatches;
    }

    boolean hasUnsupportedConditionalHeaders(HttpRequest request) {
        return request.containsHeader("If-Range") || request.containsHeader("If-Match") || request.containsHeader("If-Unmodified-Since");
    }

    boolean hasSupportedEtagValidator(HttpRequest request) {
        return request.containsHeader("If-None-Match");
    }

    boolean hasSupportedLastModifiedValidator(HttpRequest request) {
        return request.containsHeader("If-Modified-Since");
    }

    boolean etagValidatorMatches(HttpRequest request, HttpCacheEntry entry) {
        ETag etag = entry.getETag();
        if (etag == null) {
            return false;
        }
        Iterator it = MessageSupport.iterateTokens((MessageHeaders)request, (String)"If-None-Match");
        while (it.hasNext()) {
            String token = (String)it.next();
            if (!"*".equals(token) && !ETag.weakCompare((ETag)etag, (ETag)ETag.parse((String)token))) continue;
            return true;
        }
        return false;
    }

    boolean lastModifiedValidatorMatches(HttpRequest request, HttpCacheEntry entry, Instant now) {
        Instant lastModified = entry.getLastModified();
        if (lastModified == null) {
            return false;
        }
        for (Header h : request.getHeaders("If-Modified-Since")) {
            Instant ifModifiedSince = DateUtils.parseStandardDate((String)h.getValue());
            if (ifModifiedSince == null || !ifModifiedSince.isAfter(now) && !lastModified.isAfter(ifModifiedSince)) continue;
            return false;
        }
        return true;
    }

    public boolean isSuitableIfError(RequestCacheControl requestCacheControl, ResponseCacheControl responseCacheControl, HttpCacheEntry entry, Instant now) {
        if (requestCacheControl.getStaleIfError() > 0L || responseCacheControl.getStaleIfError() > 0L) {
            long stale;
            TimeValue currentAge = this.validityStrategy.getCurrentAge(entry, now);
            TimeValue freshnessLifetime = this.validityStrategy.getFreshnessLifetime(responseCacheControl, entry);
            if (requestCacheControl.getMinFresh() > 0L && requestCacheControl.getMinFresh() < freshnessLifetime.toSeconds()) {
                return false;
            }
            long l = stale = currentAge.compareTo(freshnessLifetime) > 0 ? currentAge.toSeconds() - freshnessLifetime.toSeconds() : 0L;
            if (requestCacheControl.getStaleIfError() > 0L && stale < requestCacheControl.getStaleIfError()) {
                return true;
            }
            if (responseCacheControl.getStaleIfError() > 0L && stale < responseCacheControl.getStaleIfError()) {
                return true;
            }
        }
        return this.staleifError && requestCacheControl.getStaleIfError() == -1L && responseCacheControl.getStaleIfError() == -1L;
    }
}

