#include <sys/types.h>
#include <sys/socket.h>
#include <linux/if.h>
#include <linux/if_ether.h>
#include <netinet/in.h>
#include <sys/ioctl.h>

#include <unistd.h>
#include <stdio.h>
#include <stdlib.h>
#include <time.h>

#include "axconfig.h"
#include "version.h"
#include "listen.h"

static void display_port(char *dev)
{
	char *port;

	if ((port = ax25_config_get_name(dev)) == NULL)
		port = dev;

	printf("Port %s: ", port);
}

static void display_timestamp(void)
{
	time_t timenow;
	char *timestring;

	time(&timenow);

	timestring = ctime(&timenow);	
	timestring[24] = '\0';

	printf("[%s]\n", timestring);
}

int main(int argc, char **argv)
{
	unsigned char buffer[1500];
	int timestamp = 0;
	int hexdump   = 0;
	int size;
	int s;
	char *port = NULL, *dev = NULL;
	struct sockaddr sa;
	int asize = sizeof(sa);
	struct ifreq ifr;
	int proto = ETH_P_AX25;

	while ((s = getopt(argc, argv, "ahp:tv")) != -1) {
		switch (s) {
			case 'a':
				proto = ETH_P_ALL;
				break;
			case 'h':
				hexdump = 1;
				break;
			case 'p':
				port = optarg;
				break;
			case 't':
				timestamp = 1;
				break;
			case 'v':
				printf("listen: %s\n", version);
				return 0;
			case ':':
				fprintf(stderr, "listen: option -p needs a port name\n");
				return 1;
			case '?':
				fprintf(stderr, "Usage: listen [-a] [-h] [-p port] [-t] [-v]\n");
				return 1;
		}
	}

	if (ax25_config_load_ports() == 0) {
		fprintf(stderr, "listen: no AX.25 port data configured\n");
		return 1;
	}

	if (port != NULL) {
		if ((dev = ax25_config_get_dev(port)) == NULL) {
			fprintf(stderr, "listen: invalid port name - %s\n", port);
			return 1;
		}
	}

	if ((s = socket(AF_INET, SOCK_PACKET, htons(proto))) == -1) {
		perror("socket");
		return 1;
	}
	
	for (;;) {
		if ((size = recvfrom(s, buffer, sizeof(buffer), 0, &sa, &asize)) == -1) {
			perror("recv");
			return 1;
		}
		
		if (dev != NULL && strcmp(dev, sa.sa_data) != 0)
			continue;
			
		if (proto == ETH_P_ALL) {
			strcpy(ifr.ifr_name, sa.sa_data);
			if (ioctl(s, SIOCGIFHWADDR, &ifr) < 0)
				perror("GIFADDR");

			if (ifr.ifr_hwaddr.sa_family == AF_AX25) {
				if (timestamp)
					display_timestamp();
				display_port(sa.sa_data);
				ki_dump(buffer, size, hexdump);
				putc('\n', stdout);
				fflush(stdout);
			}
			
			if (buffer[12] == 0x08 && buffer[13] == 0xFF) {
				if (timestamp)
					display_timestamp();
				display_port(sa.sa_data);
				bpq_dump(buffer, size, hexdump);
				putc('\n', stdout);
				fflush(stdout);
			}
		} else {
			if (timestamp)
				display_timestamp();
			display_port(sa.sa_data);
			ki_dump(buffer, size, hexdump);
			putc('\n', stdout);
			fflush(stdout);
		}
	}
}

static void ascii_dump(unsigned char *data, int length)
{
	unsigned char c;
	int  i, j;

	for (i = 0; length > 0; i += 64) {
		fprintf(stdout, "%04X  ", i);
	
		for (j = 0; j < 64 && length > 0; j++) {
			c = *data++;
			length--;
			
			if (c >= 0x20 && c < 0x7F)
				putc(c, stdout);
			else
				putc('.', stdout);
		}
		
		putc('\n', stdout);
	}
}

static void hex_dump(unsigned char *data, int length)
{
	int  i, j, length2;
	unsigned char c;
	char *data2;

	length2 = length;
	data2   = data;
	
	for (i = 0; length > 0; i += 16) {
		fprintf(stdout, "%04X  ", i);
	
		for (j = 0; j < 16; j++) {
			c = *data2++;
			length2--;
			
			if (length2 >= 0)
				printf("%2.2X ", c);
			else
				printf("   ");
		}

		printf(" | ");
				
		for (j = 0; j < 16 && length > 0; j++) {
 			c = *data++;
 			length--;
 			
			if (c >= 0x20 && c < 0x7F)
				putc(c, stdout);
			else
				putc('.', stdout);
		}
		
		putc('\n', stdout);
	}
}

void data_dump(unsigned char *data, int length, int hexdump)
{
	if (hexdump)
		hex_dump(data, length);
	else
		ascii_dump(data, length);
}

int get16(unsigned char *cp)
{
	int x;
	
	x = *cp++;
	x <<= 8;
	x |= *cp++;
	
	return(x);
}

int get32(unsigned char *cp)
{
	int x;
	
	x = *cp++;
	x <<= 8;
	x |= *cp++;
	x <<= 8;
	x |= *cp++;
	x <<= 8;
	x |= *cp;
	
	return(x);
}
